<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Http;

class Server extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'ip',
        'port',
        'app_id',
        'is_active',
        'api_key',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'api_key',
    ];

    /**
     * Boot function from Laravel.
     */
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($server) {
            // Generate a unique API key when creating a new server
            $server->api_key = $server->generateApiKey();
            
            // Default app_id for Garry's Mod if not provided
            if (empty($server->app_id)) {
                $server->app_id = '4000'; 
            }
        });
    }

    /**
     * Generate a unique API key.
     *
     * @return string
     */
    public function generateApiKey()
    {
        // Generate a 32 character uppercase hexadecimal string
        return strtoupper(Str::random(32));
    }

    /**
     * Query the Steam API for server information.
     *
     * @param string|null $steamApiKey The Steam API key to use for the query
     * @return array|null
     */
    public function queryServerInfo($steamApiKey = null)
    {
        if (!$steamApiKey) {
            $steamApiKey = config_value('steam_apikey');
        }
        
        $response = Http::get('https://api.steampowered.com/IGameServersService/GetServerList/v1/', [
            'filter' => "appid\\{$this->app_id}\\addr\\{$this->ip}:{$this->port}",
            'key' => $steamApiKey,
        ]);

        $data = $response->json();

        return isset($data['response']['servers'][0]) ? $data['response']['servers'][0] : null;
    }
    
    /**
     * Reset the API key for this server.
     *
     * @return string
     */
    public function resetApiKey()
    {
        $this->api_key = $this->generateApiKey();
        $this->save();
        
        return $this->api_key;
    }

    /**
     * The packages that are assigned to this server.
     */
    public function packages()
    {
        return $this->belongsToMany(Package::class, 'package_server');
    }

    /**
     * Get active packages for this server.
     */
    public function activePackages()
    {
        return $this->packages()->where('is_active', true);
    }

    /**
     * Get packages by type for this server.
     */
    public function getPackagesByType($type)
    {
        return $this->activePackages()
            ->where('type', $type)
            ->orderBy('sort_order')
            ->get();
    }
}